"""CLI entry point for HiveMind."""
from __future__ import annotations

import argparse
import asyncio
import logging
import sys
from pathlib import Path

from hivemind.config import Config
from hivemind.node import Node


def setup_logging(verbose: bool = False) -> None:
    level = logging.DEBUG if verbose else logging.INFO
    logging.basicConfig(
        level=level,
        format="%(asctime)s [%(levelname)s] %(name)s: %(message)s",
        datefmt="%H:%M:%S",
    )


async def run_node(args: argparse.Namespace) -> None:
    """Start the HiveMind node."""
    config = Config.load(args.config)
    base_dir = Path(args.config).parent

    node = Node(config, base_dir=base_dir)
    await node.start()

    tasks = []

    # Determine private key path for web UI
    private_key_path = base_dir / "hivemind_update_key.private"
    if not private_key_path.exists():
        # Also check parent dirs
        for p in [base_dir.parent, Path.home() / ".openclaw" / "workspace"]:
            candidate = p / "hivemind_update_key.private"
            if candidate.exists():
                private_key_path = candidate
                break

    # Start Web UI + API gateway if enabled
    if config.gateway.api.enabled:
        import uvicorn
        from hivemind.ui.web import create_web_app

        app = create_web_app(node, private_key_path=private_key_path)
        uv_config = uvicorn.Config(
            app,
            host=config.gateway.api.host,
            port=config.gateway.api.port,
            log_level="warning",
        )
        server = uvicorn.Server(uv_config)
        tasks.append(asyncio.create_task(server.serve()))
        logging.getLogger("hivemind").info(
            "Web UI: http://%s:%d", config.gateway.api.host, config.gateway.api.port
        )

    # Start Telegram gateway if enabled
    if config.gateway.telegram.enabled and config.gateway.telegram.token:
        from hivemind.gateway.telegram import TelegramGateway

        tg = TelegramGateway(config.gateway.telegram.token, node)
        tasks.append(asyncio.create_task(tg.start()))

    # Start TUI or daemon mode
    if args.daemon:
        logging.getLogger("hivemind").info("Running in daemon mode (Ctrl+C to stop)")
        try:
            await asyncio.gather(*tasks) if tasks else await asyncio.Event().wait()
        except asyncio.CancelledError:
            pass
    else:
        from hivemind.ui.tui import run_tui
        try:
            await run_tui(node)
        except (EOFError, KeyboardInterrupt):
            pass

    await node.stop()


def main():
    parser = argparse.ArgumentParser(
        prog="hivemind",
        description="🧠 HiveMind — Dezentrale P2P-AI",
    )
    parser.add_argument(
        "-c", "--config",
        default="config.yaml",
        help="Path to config file (default: config.yaml)",
    )
    parser.add_argument(
        "-v", "--verbose",
        action="store_true",
        help="Enable debug logging",
    )
    parser.add_argument(
        "-d", "--daemon",
        action="store_true",
        help="Run in daemon mode (no TUI, API/Telegram only)",
    )

    args = parser.parse_args()
    setup_logging(args.verbose)

    try:
        asyncio.run(run_node(args))
    except KeyboardInterrupt:
        print("\n👋 HiveMind stopped.")


if __name__ == "__main__":
    main()
