"""Web UI for HiveMind — Chat, Admin, Update Management."""
from __future__ import annotations

import json
import logging
import base64
from pathlib import Path
from typing import Any

from fastapi import FastAPI, WebSocket, WebSocketDisconnect, UploadFile, File, Form
from fastapi.responses import HTMLResponse, JSONResponse

log = logging.getLogger(__name__)


def create_web_app(node: Any, private_key_path: Path | None = None) -> FastAPI:
    """Create the web UI application."""
    app = FastAPI(title=f"HiveMind: {node.name}")

    # ─── Main UI ─────────────────────────────────────────────────
    @app.get("/", response_class=HTMLResponse)
    async def index():
        return HTML_APP

    # ─── API Endpoints ───────────────────────────────────────────
    @app.get("/api/status")
    async def status():
        return node.status

    @app.post("/api/chat")
    async def chat(data: dict):
        msg = data.get("message", "")
        if not msg:
            return {"response": ""}
        response = await node.chat(msg)
        return {"response": response}

    @app.post("/api/command")
    async def command(data: dict):
        cmd = data.get("command", "")
        if cmd == "clear":
            node.history.clear()
            return {"result": "History cleared."}
        elif cmd == "status":
            return {"result": json.dumps(node.status, indent=2, ensure_ascii=False)}
        elif cmd == "plugins":
            caps = [{"plugin": c.plugin, "name": c.name, "desc": c.description}
                    for c in node.plugins.all_capabilities()]
            return {"result": json.dumps(caps, indent=2, ensure_ascii=False)}
        elif cmd == "peers":
            if not node.network:
                return {"result": "Netzwerk nicht aktiviert."}
            peers = []
            for p in node.network.peers.all_peers:
                peers.append({
                    "name": p.name or p.node_id,
                    "address": p.address,
                    "online": p.online,
                    "hours_ago": round(p.hours_since_seen, 1),
                })
            return {"result": json.dumps(peers, indent=2, ensure_ascii=False)}
        elif cmd.startswith("addpeer "):
            addr = cmd.split(" ", 1)[1].strip()
            try:
                host, port = addr.rsplit(":", 1)
                if node.network:
                    node.network.peers.add_manual(host, int(port))
                    import asyncio
                    asyncio.create_task(node.network.connect_to(host, int(port)))
                    return {"result": f"Peer hinzugefügt: {addr}"}
                return {"result": "Netzwerk nicht aktiviert."}
            except Exception as e:
                return {"result": f"Fehler: {e}"}
        return {"result": f"Unbekannter Befehl: {cmd}"}

    # ─── Update Key ──────────────────────────────────────────────
    @app.get("/api/keys")
    async def get_keys():
        from hivemind.network.updater import PUBLISHER_PUBLIC_KEY
        priv = ""
        priv_path = private_key_path
        if priv_path and priv_path.exists():
            priv = priv_path.read_text().strip()
        return {
            "public_key": PUBLISHER_PUBLIC_KEY,
            "private_key_available": bool(priv),
            "private_key_path": str(priv_path) if priv_path else "",
        }

    # ─── Publish Update ──────────────────────────────────────────
    @app.post("/api/update/publish")
    async def publish_update(
        file: UploadFile = File(...),
        version: str = Form(...),
        changelog: str = Form(""),
    ):
        from hivemind.network.updater import sign_update, PUBLISHER_PUBLIC_KEY
        from hivemind.network.protocol import Message, MsgType

        # Read private key
        priv_path = private_key_path
        if not priv_path or not priv_path.exists():
            return JSONResponse({"error": "Private Key nicht gefunden."}, status_code=400)
        private_key = priv_path.read_text().strip()

        # Save uploaded zip
        updates_dir = node.base_dir / "updates"
        updates_dir.mkdir(exist_ok=True)
        zip_path = updates_dir / f"hivemind-{version}.zip"
        content = await file.read()
        zip_path.write_bytes(content)

        # Sign
        try:
            manifest = sign_update(zip_path, private_key, version, changelog)
        except Exception as e:
            return JSONResponse({"error": f"Signierung fehlgeschlagen: {e}"}, status_code=500)

        # Save manifest
        manifest_path = updates_dir / f"hivemind-{version}.json"
        manifest_path.write_text(json.dumps(manifest, indent=2))

        # Broadcast to network
        sent = 0
        if node.network:
            data_b64 = base64.b64encode(content).decode()
            msg = Message(
                type=MsgType.UPDATE_DATA,
                sender_id=node.id,
                payload={"manifest": manifest, "data_b64": data_b64},
            )
            sent = await node.network.broadcast(msg)

            # Also announce version
            await node.network.broadcast(Message(
                type=MsgType.VERSION_INFO,
                sender_id=node.id,
                payload={"version": version},
            ))

        return {
            "success": True,
            "version": version,
            "sha256": manifest["sha256"],
            "signature": manifest["signature"][:40] + "...",
            "peers_notified": sent,
        }

    # ─── WebSocket Chat ──────────────────────────────────────────
    @app.websocket("/ws/chat")
    async def ws_chat(ws: WebSocket):
        await ws.accept()
        try:
            while True:
                msg = await ws.receive_text()
                response = await node.chat(msg)
                await ws.send_text(response)
        except WebSocketDisconnect:
            pass

    return app


# ─── HTML/CSS/JS App ────────────────────────────────────────────

HTML_APP = """<!DOCTYPE html>
<html lang="de">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>🧠 HiveMind</title>
<style>
:root {
    --bg: #0d1117;
    --bg2: #161b22;
    --bg3: #21262d;
    --border: #30363d;
    --text: #e6edf3;
    --text2: #8b949e;
    --accent: #a855f7;
    --accent2: #7c3aed;
    --green: #3fb950;
    --red: #f85149;
    --yellow: #d29922;
}
* { margin: 0; padding: 0; box-sizing: border-box; }
body { background: var(--bg); color: var(--text); font-family: -apple-system, 'Segoe UI', sans-serif; height: 100vh; display: flex; }

/* Sidebar */
.sidebar { width: 260px; background: var(--bg2); border-right: 1px solid var(--border); display: flex; flex-direction: column; flex-shrink: 0; }
.sidebar-header { padding: 20px; border-bottom: 1px solid var(--border); }
.sidebar-header h1 { font-size: 18px; color: var(--accent); }
.sidebar-header .version { font-size: 12px; color: var(--text2); margin-top: 4px; }
.nav { flex: 1; padding: 12px; }
.nav-item { display: flex; align-items: center; gap: 10px; padding: 10px 14px; border-radius: 8px; cursor: pointer; color: var(--text2); transition: all .15s; font-size: 14px; }
.nav-item:hover { background: var(--bg3); color: var(--text); }
.nav-item.active { background: var(--accent2); color: white; }
.nav-item .icon { font-size: 18px; width: 24px; text-align: center; }
.nav-divider { height: 1px; background: var(--border); margin: 8px 14px; }

/* Quick Commands */
.quick-cmds { padding: 12px; border-top: 1px solid var(--border); }
.quick-cmds h3 { font-size: 11px; text-transform: uppercase; color: var(--text2); margin-bottom: 8px; letter-spacing: 1px; }
.cmd-btn { display: block; width: 100%; padding: 6px 10px; margin-bottom: 4px; background: var(--bg3); border: 1px solid var(--border); border-radius: 6px; color: var(--text2); font-size: 12px; cursor: pointer; text-align: left; transition: all .15s; }
.cmd-btn:hover { background: var(--accent2); color: white; border-color: var(--accent); }

/* Status bar */
.status-bar { padding: 12px; border-top: 1px solid var(--border); font-size: 11px; color: var(--text2); }
.status-dot { display: inline-block; width: 8px; height: 8px; border-radius: 50%; margin-right: 6px; }
.status-dot.online { background: var(--green); }
.status-dot.offline { background: var(--red); }

/* Main Content */
.main { flex: 1; display: flex; flex-direction: column; }
.page { display: none; flex: 1; flex-direction: column; }
.page.active { display: flex; }

/* Chat */
.chat-messages { flex: 1; overflow-y: auto; padding: 20px; }
.message { margin-bottom: 16px; max-width: 80%; }
.message.user { margin-left: auto; }
.message .bubble { padding: 12px 16px; border-radius: 12px; font-size: 14px; line-height: 1.6; white-space: pre-wrap; word-wrap: break-word; }
.message.user .bubble { background: var(--accent2); color: white; border-bottom-right-radius: 4px; }
.message.bot .bubble { background: var(--bg3); border-bottom-left-radius: 4px; }
.message .meta { font-size: 11px; color: var(--text2); margin-top: 4px; padding: 0 4px; }
.message.user .meta { text-align: right; }
.chat-input { padding: 16px 20px; border-top: 1px solid var(--border); display: flex; gap: 10px; }
.chat-input input { flex: 1; background: var(--bg3); border: 1px solid var(--border); border-radius: 10px; padding: 12px 16px; color: var(--text); font-size: 14px; outline: none; }
.chat-input input:focus { border-color: var(--accent); }
.chat-input button { background: var(--accent); border: none; border-radius: 10px; padding: 12px 20px; color: white; font-size: 14px; cursor: pointer; transition: background .15s; }
.chat-input button:hover { background: var(--accent2); }
.typing { padding: 8px 16px; font-size: 12px; color: var(--text2); font-style: italic; }

/* Admin Pages */
.admin-page { flex: 1; overflow-y: auto; padding: 24px; }
.admin-page h2 { font-size: 20px; margin-bottom: 16px; color: var(--accent); }
.card { background: var(--bg2); border: 1px solid var(--border); border-radius: 12px; padding: 20px; margin-bottom: 16px; }
.card h3 { font-size: 14px; color: var(--text2); margin-bottom: 12px; text-transform: uppercase; letter-spacing: 1px; }
.stat { display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid var(--border); font-size: 14px; }
.stat:last-child { border-bottom: none; }
.stat .label { color: var(--text2); }
.stat .value { font-weight: 600; }

/* Peers */
.peer-item { display: flex; align-items: center; gap: 12px; padding: 10px; border-bottom: 1px solid var(--border); }
.peer-item:last-child { border-bottom: none; }
.peer-status { width: 10px; height: 10px; border-radius: 50%; }
.peer-status.on { background: var(--green); }
.peer-status.off { background: var(--red); }
.peer-info { flex: 1; }
.peer-name { font-size: 14px; font-weight: 500; }
.peer-addr { font-size: 12px; color: var(--text2); }

/* Keys */
.key-display { background: var(--bg); border: 1px solid var(--border); border-radius: 8px; padding: 12px; font-family: 'Fira Code', monospace; font-size: 13px; word-break: break-all; color: var(--yellow); margin: 8px 0; }

/* Update Form */
.form-group { margin-bottom: 16px; }
.form-group label { display: block; font-size: 13px; color: var(--text2); margin-bottom: 6px; }
.form-group input, .form-group textarea { width: 100%; background: var(--bg); border: 1px solid var(--border); border-radius: 8px; padding: 10px 14px; color: var(--text); font-size: 14px; outline: none; }
.form-group input:focus, .form-group textarea:focus { border-color: var(--accent); }
.form-group textarea { min-height: 80px; resize: vertical; font-family: inherit; }
.btn { background: var(--accent); border: none; border-radius: 8px; padding: 10px 20px; color: white; font-size: 14px; cursor: pointer; transition: background .15s; }
.btn:hover { background: var(--accent2); }
.btn:disabled { opacity: 0.5; cursor: not-allowed; }
.btn-danger { background: var(--red); }
.result-box { background: var(--bg); border: 1px solid var(--border); border-radius: 8px; padding: 14px; font-family: monospace; font-size: 13px; white-space: pre-wrap; margin-top: 12px; max-height: 300px; overflow-y: auto; display: none; }

/* Add peer input */
.inline-form { display: flex; gap: 8px; margin-top: 12px; }
.inline-form input { flex: 1; background: var(--bg); border: 1px solid var(--border); border-radius: 8px; padding: 8px 12px; color: var(--text); font-size: 13px; outline: none; }
</style>
</head>
<body>

<div class="sidebar">
    <div class="sidebar-header">
        <h1>🧠 HiveMind</h1>
        <div class="version" id="version">v0.1.0</div>
    </div>
    <div class="nav">
        <div class="nav-item active" onclick="showPage('chat')">
            <span class="icon">💬</span> Chat
        </div>
        <div class="nav-item" onclick="showPage('status')">
            <span class="icon">📊</span> Status
        </div>
        <div class="nav-item" onclick="showPage('peers')">
            <span class="icon">🌐</span> Peers
        </div>
        <div class="nav-divider"></div>
        <div class="nav-item" onclick="showPage('keys')">
            <span class="icon">🔑</span> Schlüssel
        </div>
        <div class="nav-item" onclick="showPage('update')">
            <span class="icon">🚀</span> Update senden
        </div>
    </div>
    <div class="quick-cmds">
        <h3>Schnellbefehle</h3>
        <button class="cmd-btn" onclick="runCmd('status')">📊 Status</button>
        <button class="cmd-btn" onclick="runCmd('plugins')">🔌 Plugins</button>
        <button class="cmd-btn" onclick="runCmd('peers')">🌐 Peers</button>
        <button class="cmd-btn" onclick="runCmd('clear')">🗑️ Chat leeren</button>
    </div>
    <div class="status-bar" id="statusBar">
        <span class="status-dot offline" id="statusDot"></span>
        <span id="statusText">Verbinde...</span>
    </div>
</div>

<div class="main">
    <!-- Chat Page -->
    <div class="page active" id="page-chat">
        <div class="chat-messages" id="chatMessages"></div>
        <div class="typing" id="typingIndicator" style="display:none">🧠 Denkt nach...</div>
        <div class="chat-input">
            <input type="text" id="chatInput" placeholder="Nachricht eingeben..." onkeydown="if(event.key==='Enter')sendChat()">
            <button onclick="sendChat()">Senden</button>
        </div>
    </div>

    <!-- Status Page -->
    <div class="page" id="page-status">
        <div class="admin-page">
            <h2>📊 Node Status</h2>
            <div class="card" id="statusCard">Laden...</div>
        </div>
    </div>

    <!-- Peers Page -->
    <div class="page" id="page-peers">
        <div class="admin-page">
            <h2>🌐 Netzwerk & Peers</h2>
            <div class="card">
                <h3>Verbundene Peers</h3>
                <div id="peersList">Laden...</div>
                <div class="inline-form">
                    <input type="text" id="addPeerInput" placeholder="host:port">
                    <button class="btn" onclick="addPeer()">+ Peer</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Keys Page -->
    <div class="page" id="page-keys">
        <div class="admin-page">
            <h2>🔑 Update-Schlüssel</h2>
            <div class="card">
                <h3>Public Key (im Client eingebettet)</h3>
                <div class="key-display" id="publicKey">Laden...</div>
                <p style="font-size:12px;color:var(--text2)">Dieser Schlüssel verifiziert, dass Updates von dir stammen.</p>
            </div>
            <div class="card">
                <h3>Private Key</h3>
                <div id="privateKeyStatus">Laden...</div>
                <p style="font-size:12px;color:var(--text2);margin-top:8px">⚠️ Der Private Key verlässt niemals diesen Rechner.</p>
            </div>
        </div>
    </div>

    <!-- Update Page -->
    <div class="page" id="page-update">
        <div class="admin-page">
            <h2>🚀 Update veröffentlichen</h2>
            <div class="card">
                <h3>Neues Update signieren & verteilen</h3>
                <div class="form-group">
                    <label>Version</label>
                    <input type="text" id="updateVersion" placeholder="z.B. 0.2.0">
                </div>
                <div class="form-group">
                    <label>Update-Paket (.zip)</label>
                    <input type="file" id="updateFile" accept=".zip">
                </div>
                <div class="form-group">
                    <label>Changelog</label>
                    <textarea id="updateChangelog" placeholder="Was ist neu..."></textarea>
                </div>
                <button class="btn" id="publishBtn" onclick="publishUpdate()">🔏 Signieren & Senden</button>
                <div class="result-box" id="updateResult"></div>
            </div>
        </div>
    </div>
</div>

<script>
// ─── State ──────────────────────────────────────────────────────
let ws = null;
const API = '';

// ─── Navigation ─────────────────────────────────────────────────
function showPage(id) {
    document.querySelectorAll('.page').forEach(p => p.classList.remove('active'));
    document.querySelectorAll('.nav-item').forEach(n => n.classList.remove('active'));
    document.getElementById('page-' + id).classList.add('active');
    event.currentTarget.classList.add('active');

    if (id === 'status') loadStatus();
    if (id === 'peers') loadPeers();
    if (id === 'keys') loadKeys();
}

// ─── Chat ───────────────────────────────────────────────────────
function addMessage(text, role) {
    const div = document.createElement('div');
    div.className = 'message ' + role;
    const time = new Date().toLocaleTimeString('de-DE', {hour:'2-digit',minute:'2-digit'});
    div.innerHTML = `<div class="bubble">${escapeHtml(text)}</div><div class="meta">${time}</div>`;
    document.getElementById('chatMessages').appendChild(div);
    div.scrollIntoView({behavior: 'smooth'});
}

async function sendChat() {
    const input = document.getElementById('chatInput');
    const msg = input.value.trim();
    if (!msg) return;
    input.value = '';
    addMessage(msg, 'user');

    document.getElementById('typingIndicator').style.display = 'block';
    try {
        const res = await fetch(API + '/api/chat', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({message: msg})
        });
        const data = await res.json();
        addMessage(data.response, 'bot');
    } catch(e) {
        addMessage('⚠️ Fehler: ' + e.message, 'bot');
    }
    document.getElementById('typingIndicator').style.display = 'none';
}

// ─── Commands ───────────────────────────────────────────────────
async function runCmd(cmd) {
    try {
        const res = await fetch(API + '/api/command', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({command: cmd})
        });
        const data = await res.json();
        if (cmd === 'clear') {
            document.getElementById('chatMessages').innerHTML = '';
            return;
        }
        showPage('chat');
        document.querySelector('.nav-item').click();
        addMessage('/' + cmd + '\\n\\n' + data.result, 'bot');
    } catch(e) {
        console.error(e);
    }
}

// ─── Status ─────────────────────────────────────────────────────
async function loadStatus() {
    try {
        const res = await fetch(API + '/api/status');
        const s = await res.json();
        document.getElementById('version').textContent = 'v' + (s.version || '0.1.0');
        let html = '';
        const entries = [
            ['Node ID', s.id],
            ['Name', s.name],
            ['Version', s.version],
            ['Modell', s.model_loaded ? '✅ Geladen' : '❌ Nicht geladen'],
            ['Plugins', (s.plugins||[]).join(', ')],
            ['Cache', s.cache_size + ' Einträge'],
            ['Chat-Verlauf', s.history_length + ' Nachrichten'],
        ];
        if (s.network) {
            entries.push(['P2P Port', s.network.listening]);
            entries.push(['Verbundene Peers', s.network.connected]);
            entries.push(['Bekannte Peers', s.network.known_peers]);
            entries.push(['Online Peers', s.network.online_peers]);
        }
        html = entries.map(([l,v]) => `<div class="stat"><span class="label">${l}</span><span class="value">${v}</span></div>`).join('');
        document.getElementById('statusCard').innerHTML = html;

        // Update status bar
        document.getElementById('statusDot').className = 'status-dot online';
        const netInfo = s.network ? ` | ${s.network.peers}` : '';
        document.getElementById('statusText').textContent = s.name + netInfo;
    } catch(e) {
        document.getElementById('statusCard').textContent = 'Fehler: ' + e.message;
    }
}

// ─── Peers ──────────────────────────────────────────────────────
async function loadPeers() {
    try {
        const res = await fetch(API + '/api/command', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({command: 'peers'})
        });
        const data = await res.json();
        let peers;
        try { peers = JSON.parse(data.result); } catch(e) { 
            document.getElementById('peersList').innerHTML = '<div style="color:var(--text2);padding:10px">' + escapeHtml(data.result) + '</div>';
            return;
        }
        if (!peers.length) {
            document.getElementById('peersList').innerHTML = '<div style="color:var(--text2);padding:10px">Keine Peers bekannt. Füge einen hinzu!</div>';
            return;
        }
        document.getElementById('peersList').innerHTML = peers.map(p => `
            <div class="peer-item">
                <div class="peer-status ${p.online?'on':'off'}"></div>
                <div class="peer-info">
                    <div class="peer-name">${escapeHtml(p.name)}</div>
                    <div class="peer-addr">${escapeHtml(p.address)} · ${p.online?'Online':p.hours_ago+'h her'}</div>
                </div>
            </div>
        `).join('');
    } catch(e) {
        document.getElementById('peersList').textContent = 'Fehler: ' + e.message;
    }
}

async function addPeer() {
    const input = document.getElementById('addPeerInput');
    const addr = input.value.trim();
    if (!addr) return;
    await fetch(API + '/api/command', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({command: 'addpeer ' + addr})
    });
    input.value = '';
    setTimeout(loadPeers, 1000);
}

// ─── Keys ───────────────────────────────────────────────────────
async function loadKeys() {
    try {
        const res = await fetch(API + '/api/keys');
        const data = await res.json();
        document.getElementById('publicKey').textContent = data.public_key || 'Nicht konfiguriert';
        if (data.private_key_available) {
            document.getElementById('privateKeyStatus').innerHTML = '<span style="color:var(--green)">✅ Verfügbar</span><br><span style="font-size:12px;color:var(--text2)">Pfad: ' + escapeHtml(data.private_key_path) + '</span>';
        } else {
            document.getElementById('privateKeyStatus').innerHTML = '<span style="color:var(--red)">❌ Nicht gefunden</span><br><span style="font-size:12px;color:var(--text2)">Lege den Private Key ab unter: hivemind_update_key.private</span>';
        }
    } catch(e) {
        document.getElementById('publicKey').textContent = 'Fehler: ' + e.message;
    }
}

// ─── Publish Update ─────────────────────────────────────────────
async function publishUpdate() {
    const version = document.getElementById('updateVersion').value.trim();
    const fileInput = document.getElementById('updateFile');
    const changelog = document.getElementById('updateChangelog').value.trim();
    const resultBox = document.getElementById('updateResult');
    const btn = document.getElementById('publishBtn');

    if (!version) { alert('Version eingeben!'); return; }
    if (!fileInput.files.length) { alert('ZIP-Datei auswählen!'); return; }

    btn.disabled = true;
    btn.textContent = '⏳ Signiere & sende...';
    resultBox.style.display = 'block';
    resultBox.textContent = 'Wird verarbeitet...';

    const form = new FormData();
    form.append('file', fileInput.files[0]);
    form.append('version', version);
    form.append('changelog', changelog);

    try {
        const res = await fetch(API + '/api/update/publish', { method: 'POST', body: form });
        const data = await res.json();
        if (data.error) {
            resultBox.textContent = '❌ Fehler: ' + data.error;
            resultBox.style.color = 'var(--red)';
        } else {
            resultBox.style.color = 'var(--green)';
            resultBox.textContent = `✅ Update v${data.version} veröffentlicht!

SHA256:    ${data.sha256}
Signatur:  ${data.signature}
Peers informiert: ${data.peers_notified}`;
        }
    } catch(e) {
        resultBox.textContent = '❌ Fehler: ' + e.message;
        resultBox.style.color = 'var(--red)';
    }

    btn.disabled = false;
    btn.textContent = '🔏 Signieren & Senden';
}

// ─── Helpers ────────────────────────────────────────────────────
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// ─── Init ───────────────────────────────────────────────────────
loadStatus();
document.getElementById('chatInput').focus();
setInterval(loadStatus, 30000);
</script>
</body>
</html>"""
