"""HiveMind Node — the core unit that ties everything together."""
from __future__ import annotations

import asyncio
import logging
import uuid
from pathlib import Path

from hivemind import __version__
from hivemind.config import Config
from hivemind.model import Model
from hivemind.cache import SemanticCache
from hivemind.plugins import PluginManager

log = logging.getLogger(__name__)


class Node:
    """A single HiveMind node — local AI + cache + plugins."""

    def __init__(self, config: Config, base_dir: Path | None = None):
        self.config = config
        self.base_dir = base_dir or Path(".")
        self.id = str(uuid.uuid4())[:8]
        self.name = config.node.name or f"node-{self.id}"
        self.version = __version__

        # Core components
        self.model = Model(config.model)
        self.cache = SemanticCache(config.cache, cache_dir=self.base_dir / "cache")
        self.plugins = PluginManager(node=self)
        self.network = None

        # Conversation history
        self.history: list[dict] = []
        self.max_history = 50

        self._running = False

    async def start(self) -> None:
        """Initialize the node — load model, plugins, and network."""
        log.info("Starting node: %s (%s) v%s", self.name, self.id, self.version)

        # Load model if configured
        if self.config.model.path:
            try:
                self.model.load()
            except FileNotFoundError as e:
                log.error("Model not found: %s", e)
                log.info("Node will run without local model (cache + plugins only)")
        else:
            log.info("No model configured — running in plugin-only mode")

        # Load plugins
        await self.plugins.load(
            self.config.plugins.get("enabled", []),
            self.config.plugins.get("directory", "./plugins"),
        )

        # Start P2P network if enabled
        if self.config.network.enabled:
            from hivemind.network.peerlist import PeerList
            from hivemind.network.peer import P2PNetwork
            from hivemind.network.updater import AutoUpdater

            peer_list = PeerList(
                path=self.base_dir / "peers.json",
                own_id=self.id,
            )

            # Add bootstrap nodes from config
            for addr in self.config.network.bootstrap_nodes:
                parts = addr.split(":")
                if len(parts) == 2:
                    peer_list.add_manual(parts[0], int(parts[1]))

            self.network = P2PNetwork(
                node=self,
                peer_list=peer_list,
                listen_port=self.config.network.listen_port,
            )

            # Setup auto-updater
            updater = AutoUpdater(self, self.base_dir)
            updater.register_handlers(self.network)

            await self.network.start()
            log.info("P2P network started on port %d", self.config.network.listen_port)

        self._running = True
        net_status = ""
        if self.network:
            net_status = f" | Network: port {self.config.network.listen_port}"
        log.info(
            "Node ready: %s | Model: %s | Plugins: %s | Cache: %d entries%s",
            self.name,
            "loaded" if self.model.loaded else "none",
            ", ".join(self.plugins.loaded) or "none",
            self.cache.size,
            net_status,
        )

    async def chat(self, user_message: str) -> str:
        """Process a user message and return a response.
        
        Flow:
        1. Check cache for similar query
        2. If miss: generate via local model
        3. Store in cache
        4. Return response
        """
        # 1. Cache lookup
        cached = self.cache.lookup(user_message)
        if cached:
            log.info("Cache hit for: %s", user_message[:50])
            return cached

        # 2. Add to history
        self.history.append({"role": "user", "content": user_message})

        # Trim history if too long
        if len(self.history) > self.max_history:
            self.history = self.history[-self.max_history:]

        # 3. Generate response
        chat_plugin = self.plugins.get("chat")
        if chat_plugin:
            response = await chat_plugin.capabilities[0].handler(
                messages=list(self.history)
            )
        elif self.model.loaded:
            response = self.model.generate(self.history)
        else:
            response = (
                "⚠️ Kein Modell geladen und kein Chat-Plugin verfügbar.\n"
                "Konfiguriere model.path in config.yaml oder installiere ein Plugin."
            )

        # 4. Store in history and cache
        self.history.append({"role": "assistant", "content": response})
        self.cache.store(user_message, response)

        return response

    async def stop(self) -> None:
        """Shutdown the node gracefully."""
        log.info("Stopping node: %s", self.name)
        if self.network:
            await self.network.stop()
        await self.plugins.shutdown_all()
        self._running = False

    @property
    def status(self) -> dict:
        s = {
            "id": self.id,
            "name": self.name,
            "version": self.version,
            "model_loaded": self.model.loaded,
            "plugins": self.plugins.loaded,
            "cache_size": self.cache.size,
            "history_length": len(self.history),
            "running": self._running,
        }
        if self.network:
            s["network"] = self.network.status
        return s
