"""Plugin base class and manager."""
from __future__ import annotations

import logging
import importlib
from pathlib import Path
from typing import Any, Callable
from dataclasses import dataclass, field

log = logging.getLogger(__name__)


@dataclass
class Capability:
    """A single capability provided by a plugin."""
    name: str
    description: str
    handler: Callable
    plugin: str


def capability(description: str):
    """Decorator to mark a method as a plugin capability."""
    def decorator(func):
        func._capability = description
        return func
    return decorator


class Plugin:
    """Base class for all HiveMind plugins."""
    name: str = "unnamed"
    version: str = "0.1.0"
    description: str = ""

    def __init__(self, node: Any = None):
        self.node = node
        self._capabilities: list[Capability] = []
        self._discover_capabilities()

    def _discover_capabilities(self) -> None:
        """Find all methods decorated with @capability."""
        for attr_name in dir(self):
            attr = getattr(self, attr_name, None)
            if callable(attr) and hasattr(attr, "_capability"):
                self._capabilities.append(Capability(
                    name=attr_name,
                    description=attr._capability,
                    handler=attr,
                    plugin=self.name,
                ))

    @property
    def capabilities(self) -> list[Capability]:
        return self._capabilities

    async def initialize(self) -> None:
        """Called when plugin is loaded. Override for setup."""
        pass

    async def shutdown(self) -> None:
        """Called when plugin is unloaded. Override for cleanup."""
        pass


class PluginManager:
    """Discovers, loads and manages plugins."""

    # Built-in plugin modules
    BUILTIN = {
        "chat": "hivemind.plugins.chat",
        "web_search": "hivemind.plugins.web_search",
    }

    def __init__(self, node: Any = None):
        self.node = node
        self._plugins: dict[str, Plugin] = {}

    async def load(self, plugin_names: list[str], plugin_dir: str = "./plugins") -> None:
        """Load plugins by name."""
        for name in plugin_names:
            try:
                plugin = self._load_one(name, plugin_dir)
                if plugin:
                    await plugin.initialize()
                    self._plugins[name] = plugin
                    caps = ", ".join(c.name for c in plugin.capabilities)
                    log.info("Plugin loaded: %s [%s]", name, caps)
            except Exception as e:
                log.error("Failed to load plugin '%s': %s", name, e)

    def _load_one(self, name: str, plugin_dir: str) -> Plugin | None:
        """Load a single plugin by name."""
        # Try built-in first
        if name in self.BUILTIN:
            mod = importlib.import_module(self.BUILTIN[name])
            cls = getattr(mod, "PLUGIN_CLASS")
            return cls(node=self.node)

        # Try community plugin directory
        plugin_path = Path(plugin_dir) / name / "__init__.py"
        if plugin_path.exists():
            import importlib.util as ilu
            spec = ilu.spec_from_file_location(f"plugins.{name}", plugin_path)
            mod = ilu.module_from_spec(spec)
            spec.loader.exec_module(mod)
            cls = getattr(mod, "PLUGIN_CLASS")
            return cls(node=self.node)

        log.warning("Plugin not found: %s", name)
        return None

    def get(self, name: str) -> Plugin | None:
        return self._plugins.get(name)

    def all_capabilities(self) -> list[Capability]:
        """Get all capabilities from all loaded plugins."""
        caps = []
        for plugin in self._plugins.values():
            caps.extend(plugin.capabilities)
        return caps

    def capabilities_prompt(self) -> str:
        """Generate a system prompt section describing available capabilities."""
        lines = ["Available tools:"]
        for cap in self.all_capabilities():
            lines.append(f"- {cap.plugin}.{cap.name}: {cap.description}")
        return "\n".join(lines)

    async def execute(self, plugin_name: str, capability_name: str, **kwargs) -> Any:
        """Execute a specific capability."""
        plugin = self._plugins.get(plugin_name)
        if not plugin:
            raise ValueError(f"Plugin not loaded: {plugin_name}")
        
        for cap in plugin.capabilities:
            if cap.name == capability_name:
                return await cap.handler(**kwargs)
        
        raise ValueError(f"Capability not found: {plugin_name}.{capability_name}")

    async def shutdown_all(self) -> None:
        for plugin in self._plugins.values():
            await plugin.shutdown()

    @property
    def loaded(self) -> list[str]:
        return list(self._plugins.keys())
