#!/usr/bin/env python3
"""
🧠 HiveMind — Geführte Installation
Funktioniert auf Linux und Windows.
"""

import os
import sys
import json
import shutil
import platform
import subprocess
import ssl
import urllib.request
import urllib.error
from pathlib import Path

# ─── Farben (Terminal) ──────────────────────────────────────────────
if platform.system() == "Windows":
    os.system("")  # Enable ANSI on Windows 10+

BOLD = "\033[1m"
DIM = "\033[2m"
GREEN = "\033[32m"
CYAN = "\033[36m"
YELLOW = "\033[33m"
RED = "\033[31m"
MAGENTA = "\033[35m"
RESET = "\033[0m"


# ─── Verfügbare Modelle ─────────────────────────────────────────────
MODELS = [
    {
        "id": "qwen2.5-0.5b",
        "name": "Qwen 2.5 0.5B (Winzig)",
        "size": "0.4 GB",
        "ram": "1 GB",
        "speed": "⚡⚡⚡ Sehr schnell",
        "quality": "★☆☆☆☆ Einfache Aufgaben",
        "url": "https://huggingface.co/bartowski/Qwen2.5-0.5B-Instruct-GGUF/resolve/main/Qwen2.5-0.5B-Instruct-Q4_K_M.gguf",
        "filename": "Qwen2.5-0.5B-Instruct-Q4_K_M.gguf",
    },
    {
        "id": "qwen2.5-1.5b",
        "name": "Qwen 2.5 1.5B (Klein)",
        "size": "1.0 GB",
        "ram": "2 GB",
        "speed": "⚡⚡⚡ Schnell",
        "quality": "★★☆☆☆ Alltägliche Aufgaben",
        "url": "https://huggingface.co/bartowski/Qwen2.5-1.5B-Instruct-GGUF/resolve/main/Qwen2.5-1.5B-Instruct-Q4_K_M.gguf",
        "filename": "Qwen2.5-1.5B-Instruct-Q4_K_M.gguf",
    },
    {
        "id": "qwen2.5-3b",
        "name": "Qwen 2.5 3B (Mittel)",
        "size": "1.8 GB",
        "ram": "4 GB",
        "speed": "⚡⚡ Mittel",
        "quality": "★★★☆☆ Gute Qualität",
        "url": "https://huggingface.co/bartowski/Qwen2.5-3B-Instruct-GGUF/resolve/main/Qwen2.5-3B-Instruct-Q4_K_M.gguf",
        "filename": "Qwen2.5-3B-Instruct-Q4_K_M.gguf",
    },
    {
        "id": "qwen2.5-7b",
        "name": "Qwen 2.5 7B (Groß)",
        "size": "4.4 GB",
        "ram": "8 GB",
        "speed": "⚡ Langsamer",
        "quality": "★★★★☆ Sehr gute Qualität",
        "url": "https://huggingface.co/bartowski/Qwen2.5-7B-Instruct-GGUF/resolve/main/Qwen2.5-7B-Instruct-Q4_K_M.gguf",
        "filename": "Qwen2.5-7B-Instruct-Q4_K_M.gguf",
    },
    {
        "id": "llama3.2-3b",
        "name": "Llama 3.2 3B (Meta)",
        "size": "2.0 GB",
        "ram": "4 GB",
        "speed": "⚡⚡ Mittel",
        "quality": "★★★☆☆ Gut für Englisch",
        "url": "https://huggingface.co/bartowski/Llama-3.2-3B-Instruct-GGUF/resolve/main/Llama-3.2-3B-Instruct-Q4_K_M.gguf",
        "filename": "Llama-3.2-3B-Instruct-Q4_K_M.gguf",
    },
    {
        "id": "none",
        "name": "Kein Modell (später manuell)",
        "size": "-",
        "ram": "-",
        "speed": "",
        "quality": "",
        "url": None,
        "filename": None,
    },
]

MIN_PYTHON = (3, 10)


def _create_brain_icon(path: Path):
    """Erstelle ein 32x32 Brain-Icon als .ico (pure Python, keine Deps)."""
    import struct, zlib

    # 32x32 RGBA pixel data — purple brain on transparent background
    W = 32
    pixels = bytearray(W * W * 4)  # RGBA

    def put(x, y, r, g, b, a=255):
        if 0 <= x < W and 0 <= y < W:
            i = (y * W + x) * 4
            pixels[i:i+4] = bytes([r, g, b, a])

    def circle(cx, cy, rx, ry, r, g, b, a=255):
        for y in range(W):
            for x in range(W):
                dx = (x - cx) / max(rx, 0.1)
                dy = (y - cy) / max(ry, 0.1)
                if dx*dx + dy*dy <= 1.0:
                    put(x, y, r, g, b, a)

    def line_h(x1, x2, y, r, g, b, a=255):
        for x in range(x1, x2+1):
            put(x, y, r, g, b, a)

    # Brain shape — two hemispheres
    # Left hemisphere
    circle(12, 14, 9, 11, 168, 85, 247)   # main purple
    circle(10, 10, 6, 5, 147, 60, 230)    # upper left lobe
    circle(12, 19, 7, 5, 147, 60, 230)    # lower left lobe
    # Right hemisphere
    circle(20, 14, 9, 11, 168, 85, 247)
    circle(22, 10, 6, 5, 147, 60, 230)
    circle(20, 19, 7, 5, 147, 60, 230)

    # Center line (fissure)
    for y in range(5, 27):
        put(16, y, 90, 30, 160)
        put(15, y, 130, 50, 200, 128)
        put(17, y, 130, 50, 200, 128)

    # Sulci (grooves) — lighter lines
    for y in range(8, 22):
        if y % 3 == 0:
            line_h(8, 14, y, 190, 120, 255, 180)
            line_h(18, 24, y, 190, 120, 255, 180)

    # Highlight (shine)
    circle(11, 9, 3, 2, 210, 170, 255, 100)
    circle(21, 9, 3, 2, 210, 170, 255, 100)

    # Brain stem
    circle(16, 26, 3, 3, 130, 50, 200)
    circle(16, 28, 2, 2, 110, 40, 180)

    # Convert RGBA to BGRA for ICO format
    bgra = bytearray(len(pixels))
    for i in range(0, len(pixels), 4):
        bgra[i] = pixels[i+2]      # B
        bgra[i+1] = pixels[i+1]    # G
        bgra[i+2] = pixels[i]      # R
        bgra[i+3] = pixels[i+3]    # A

    # Build ICO file
    # ICO header: reserved(2) + type(2) + count(2)
    ico_header = struct.pack('<HHH', 0, 1, 1)
    # Image entry: w, h, colors, reserved, planes, bpp, size, offset
    bmp_header_size = 40
    pixel_data_size = W * W * 4
    image_size = bmp_header_size + pixel_data_size
    ico_entry = struct.pack('<BBBBHHII', W, W, 0, 0, 1, 32, image_size, 22)
    # BMP info header (BITMAPINFOHEADER)
    bmp_header = struct.pack('<IiiHHIIiiII',
        40,     # size
        W,      # width
        W * 2,  # height (doubled for ICO: image + mask)
        1,      # planes
        32,     # bpp
        0,      # compression
        pixel_data_size,
        0, 0, 0, 0
    )
    # ICO BMPs are stored bottom-up
    flipped = bytearray()
    for y in range(W - 1, -1, -1):
        row_start = y * W * 4
        flipped.extend(bgra[row_start:row_start + W * 4])

    path.write_bytes(ico_header + ico_entry + bmp_header + bytes(flipped))


def banner():
    print(f"""
{MAGENTA}{BOLD}╔══════════════════════════════════════════════╗
║          🧠 HiveMind — Installation          ║
║       Dezentrale P2P-AI für alle              ║
╚══════════════════════════════════════════════╝{RESET}
""")


def check_python():
    """Prüfe Python-Version."""
    v = sys.version_info
    print(f"  Python: {v.major}.{v.minor}.{v.micro}", end="")
    if v >= MIN_PYTHON:
        print(f"  {GREEN}✓{RESET}")
        return True
    else:
        print(f"  {RED}✗ (mindestens {MIN_PYTHON[0]}.{MIN_PYTHON[1]} benötigt){RESET}")
        return False


def check_disk(install_dir: Path):
    """Prüfe verfügbaren Speicherplatz."""
    try:
        usage = shutil.disk_usage(install_dir.parent)
        free_gb = usage.free / (1024 ** 3)
        print(f"  Speicherplatz: {free_gb:.1f} GB frei", end="")
        if free_gb >= 2:
            print(f"  {GREEN}✓{RESET}")
        else:
            print(f"  {YELLOW}⚠ Wenig Platz{RESET}")
        return free_gb
    except Exception:
        print(f"  Speicherplatz: {DIM}nicht ermittelbar{RESET}")
        return 999


def check_ram():
    """Prüfe verfügbaren RAM."""
    try:
        if platform.system() == "Windows":
            import ctypes
            class MEMORYSTATUSEX(ctypes.Structure):
                _fields_ = [
                    ("dwLength", ctypes.c_ulong),
                    ("dwMemoryLoad", ctypes.c_ulong),
                    ("ullTotalPhys", ctypes.c_ulonglong),
                    ("ullAvailPhys", ctypes.c_ulonglong),
                    ("ullTotalPageFile", ctypes.c_ulonglong),
                    ("ullAvailPageFile", ctypes.c_ulonglong),
                    ("ullTotalVirtual", ctypes.c_ulonglong),
                    ("ullAvailVirtual", ctypes.c_ulonglong),
                    ("ullAvailExtendedVirtual", ctypes.c_ulonglong),
                ]
            stat = MEMORYSTATUSEX()
            stat.dwLength = ctypes.sizeof(stat)
            ctypes.windll.kernel32.GlobalMemoryStatusEx(ctypes.byref(stat))
            total_gb = stat.ullTotalPhys / (1024 ** 3)
        else:
            with open("/proc/meminfo") as f:
                for line in f:
                    if line.startswith("MemTotal:"):
                        total_gb = int(line.split()[1]) / (1024 ** 2)
                        break
        print(f"  RAM: {total_gb:.1f} GB", end="")
        print(f"  {GREEN}✓{RESET}" if total_gb >= 2 else f"  {YELLOW}⚠{RESET}")
        return total_gb
    except Exception:
        print(f"  RAM: {DIM}nicht ermittelbar{RESET}")
        return 16


def get_default_install_dir():
    """Standard-Installationsverzeichnis."""
    if platform.system() == "Windows":
        base = Path(os.environ.get("LOCALAPPDATA", Path.home() / "AppData" / "Local"))
        return base / "HiveMind"
    else:
        return Path.home() / ".hivemind"


def ask(prompt, default=None):
    """Frage mit optionalem Default."""
    if default:
        prompt = f"{prompt} [{default}]"
    try:
        answer = input(f"{CYAN}  → {prompt}: {RESET}").strip()
    except (EOFError, KeyboardInterrupt):
        print(f"\n{RED}  Abgebrochen.{RESET}")
        sys.exit(1)
    return answer if answer else default


def ask_choice(prompt, options, default=1):
    """Auswahl aus nummerierten Optionen."""
    print(f"\n{BOLD}  {prompt}{RESET}\n")
    for i, opt in enumerate(options, 1):
        marker = f"{GREEN}→{RESET}" if i == default else " "
        print(f"  {marker} {BOLD}{i}{RESET}) {opt}")
    while True:
        try:
            answer = input(f"\n{CYAN}  → Auswahl [1-{len(options)}, Standard={default}]: {RESET}").strip()
        except (EOFError, KeyboardInterrupt):
            print(f"\n{RED}  Abgebrochen.{RESET}")
            sys.exit(1)
        if not answer:
            return default - 1
        try:
            idx = int(answer) - 1
            if 0 <= idx < len(options):
                return idx
        except ValueError:
            pass
        print(f"  {RED}Ungültige Eingabe.{RESET}")


def download_file(url, dest_path, label=""):
    """Download mit Fortschrittsanzeige. Nutzt PowerShell auf Windows als Fallback."""
    print(f"\n  {BOLD}Downloading: {label}{RESET}")
    print(f"  {DIM}{url}{RESET}\n")

    dest_path.parent.mkdir(parents=True, exist_ok=True)

    # On Windows, prefer PowerShell (handles SSL/redirects via system certs)
    if platform.system() == "Windows":
        try:
            ps_cmd = (
                f"[Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12; "
                f"$ProgressPreference = 'SilentlyContinue'; "
                f"Invoke-WebRequest -Uri '{url}' -OutFile '{dest_path}' -UseBasicParsing"
            )
            print(f"  Lade herunter (PowerShell)...")
            result = subprocess.run(
                ["powershell", "-Command", ps_cmd],
                capture_output=True, text=True, timeout=1800
            )
            if result.returncode == 0 and dest_path.exists() and dest_path.stat().st_size > 1000:
                size_mb = dest_path.stat().st_size / (1024 * 1024)
                print(f"  {GREEN}✓ Download abgeschlossen ({size_mb:.0f} MB){RESET}")
                return True
            else:
                err = result.stderr.strip().split('\n')[-1] if result.stderr else "Unbekannt"
                print(f"  {YELLOW}PowerShell-Download fehlgeschlagen: {err}{RESET}")
                if dest_path.exists():
                    dest_path.unlink()
        except Exception as e:
            print(f"  {YELLOW}PowerShell-Fehler: {e}{RESET}")

    # Fallback: urllib
    try:
        req = urllib.request.Request(url, headers={"User-Agent": "HiveMind-Installer/1.0"})
        try:
            response_ctx = urllib.request.urlopen(req, timeout=30)
        except urllib.error.URLError:
            ctx = ssl.create_default_context()
            ctx.check_hostname = False
            ctx.verify_mode = ssl.CERT_NONE
            opener = urllib.request.build_opener(urllib.request.HTTPSHandler(context=ctx))
            response_ctx = opener.open(req, timeout=30)
        with response_ctx as response:
            total = int(response.headers.get("Content-Length", 0))
            downloaded = 0
            chunk_size = 1024 * 256  # 256KB chunks

            dest_path.parent.mkdir(parents=True, exist_ok=True)
            with open(dest_path, "wb") as f:
                while True:
                    chunk = response.read(chunk_size)
                    if not chunk:
                        break
                    f.write(chunk)
                    downloaded += len(chunk)
                    if total > 0:
                        pct = downloaded / total * 100
                        bar_len = 40
                        filled = int(bar_len * downloaded // total)
                        bar = "█" * filled + "░" * (bar_len - filled)
                        dl_mb = downloaded / (1024 * 1024)
                        total_mb = total / (1024 * 1024)
                        print(f"\r  {bar} {pct:5.1f}% ({dl_mb:.0f}/{total_mb:.0f} MB)", end="", flush=True)
                    else:
                        dl_mb = downloaded / (1024 * 1024)
                        print(f"\r  {dl_mb:.0f} MB heruntergeladen...", end="", flush=True)
            print(f"\n  {GREEN}✓ Download abgeschlossen{RESET}")
            return True
    except (urllib.error.URLError, urllib.error.HTTPError, OSError) as e:
        print(f"\n  {RED}✗ Download fehlgeschlagen: {e}{RESET}")
        if dest_path.exists():
            dest_path.unlink()
        return False


def find_system_python():
    """Finde das echte System-Python (nicht Inkscape/embedded)."""
    current = sys.executable
    # Check if current python is from a known embedded location
    embedded_markers = ["inkscape", "blender", "gimp", "embedded"]
    current_lower = current.lower()
    is_embedded = any(m in current_lower for m in embedded_markers)

    if not is_embedded:
        return current

    # Try to find system Python
    if platform.system() == "Windows":
        candidates = [
            Path(os.environ.get("LOCALAPPDATA", "")) / "Programs" / "Python" / "Python312" / "python.exe",
            Path("C:/Program Files/Python312/python.exe"),
            Path("C:/Program Files/Python311/python.exe"),
            Path(os.environ.get("LOCALAPPDATA", "")) / "Programs" / "Python" / "Python311" / "python.exe",
        ]
        for c in candidates:
            if c.exists():
                print(f"  {YELLOW}⚠ Eingebettetes Python erkannt ({Path(current).parent.name}){RESET}")
                print(f"  {DIM}Nutze stattdessen: {c}{RESET}")
                return str(c)

        # Try 'py' launcher
        try:
            result = subprocess.run(["py", "-3", "-c", "import sys; print(sys.executable)"],
                                    capture_output=True, text=True, timeout=10)
            if result.returncode == 0:
                found = result.stdout.strip()
                if found and not any(m in found.lower() for m in embedded_markers):
                    print(f"  {YELLOW}⚠ Eingebettetes Python erkannt, nutze: {found}{RESET}")
                    return found
        except Exception:
            pass

    print(f"  {YELLOW}⚠ Kein separates System-Python gefunden, nutze aktuelles.{RESET}")
    return current


def create_venv(install_dir: Path):
    """Erstelle Virtual Environment und installiere HiveMind."""
    venv_dir = install_dir / ".venv"
    python_cmd = find_system_python()

    print(f"\n  {BOLD}Erstelle Virtual Environment...{RESET}")
    print(f"  {DIM}Python: {python_cmd}{RESET}")

    result = subprocess.run(
        [python_cmd, "-m", "venv", str(venv_dir)],
        capture_output=True, text=True
    )
    if result.returncode != 0:
        print(f"  {RED}✗ venv-Erstellung fehlgeschlagen:{RESET}")
        print(f"  {result.stderr}")
        # Try with --without-pip as fallback
        print(f"  {YELLOW}Versuche Fallback (ohne pip)...{RESET}")
        result = subprocess.run(
            [python_cmd, "-m", "venv", "--without-pip", str(venv_dir)],
            capture_output=True, text=True
        )
        if result.returncode != 0:
            return False

    # Determine paths
    if platform.system() == "Windows":
        pip_path = venv_dir / "Scripts" / "pip.exe"
        python_path = venv_dir / "Scripts" / "python.exe"
    else:
        pip_path = venv_dir / "bin" / "pip"
        python_path = venv_dir / "bin" / "python"

    # Check python exists in venv
    if not python_path.exists():
        print(f"  {RED}✗ Python nicht im venv gefunden: {python_path}{RESET}")
        return False

    # Ensure pip is available
    if not pip_path.exists():
        print(f"  {DIM}Installiere pip...{RESET}")
        result = subprocess.run(
            [str(python_path), "-m", "ensurepip", "--upgrade"],
            capture_output=True, text=True
        )
        if result.returncode != 0:
            # Last resort: get-pip.py
            print(f"  {YELLOW}ensurepip fehlgeschlagen, versuche get-pip.py...{RESET}")
            getpip = install_dir / "get-pip.py"
            try:
                getpip_url = "https://bootstrap.pypa.io/get-pip.py"
                try:
                    urllib.request.urlretrieve(getpip_url, str(getpip))
                except urllib.error.URLError:
                    ctx = ssl.create_default_context()
                    ctx.check_hostname = False
                    ctx.verify_mode = ssl.CERT_NONE
                    req = urllib.request.Request(getpip_url)
                    with urllib.request.urlopen(req, context=ctx) as resp:
                        getpip.write_bytes(resp.read())
                subprocess.run([str(python_path), str(getpip)],
                               capture_output=True, text=True)
                getpip.unlink(missing_ok=True)
            except Exception as e:
                print(f"  {RED}✗ pip-Installation fehlgeschlagen: {e}{RESET}")
                return False

    # Upgrade pip
    print(f"  {DIM}Aktualisiere pip...{RESET}")
    subprocess.run(
        [str(python_path), "-m", "pip", "install", "--upgrade", "pip"],
        capture_output=True, text=True
    )

    # Install llama-cpp-python (may need compilation)
    print(f"\n  {BOLD}Installiere llama-cpp-python...{RESET}")
    print(f"  {DIM}(Dies kann einige (bis zu 30) Minuten dauern — C++ wird kompiliert){RESET}")

    env = os.environ.copy()
    env["CMAKE_ARGS"] = "-DGGML_NATIVE=OFF"

    result = subprocess.run(
        [str(python_path), "-m", "pip", "install", "llama-cpp-python>=0.3.0"],
        capture_output=True, text=True, env=env, timeout=600
    )
    if result.returncode != 0:
        print(f"  {RED}✗ llama-cpp-python Installation fehlgeschlagen{RESET}")
        print(f"  {DIM}Häufige Ursache: Fehlende Build-Tools{RESET}")
        if platform.system() == "Windows":
            print(f"  {YELLOW}→ Installiere 'Visual Studio Build Tools' von:")
            print(f"    https://visualstudio.microsoft.com/visual-cpp-build-tools/{RESET}")
        else:
            print(f"  {YELLOW}→ Installiere: sudo apt install build-essential cmake{RESET}")
        # Show last few lines of error
        err_lines = result.stderr.strip().split("\n")[-5:]
        for line in err_lines:
            print(f"  {DIM}{line}{RESET}")
        return False
    print(f"  {GREEN}✓ llama-cpp-python installiert{RESET}")

    # Install remaining dependencies
    print(f"\n  {BOLD}Installiere weitere Abhängigkeiten...{RESET}")
    deps = [
        "fastapi>=0.115.0",
        "uvicorn>=0.34.0",
        "numpy>=1.26.0",
        "pyyaml>=6.0",
        "rich>=13.0",
        "prompt-toolkit>=3.0",
        "httpx>=0.27.0",
        "beautifulsoup4>=4.12.0",
    ]
    result = subprocess.run(
        [str(python_path), "-m", "pip", "install"] + deps,
        capture_output=True, text=True, timeout=300
    )
    if result.returncode != 0:
        print(f"  {RED}✗ Abhängigkeiten fehlgeschlagen{RESET}")
        return False
    print(f"  {GREEN}✓ Alle Abhängigkeiten installiert{RESET}")

    # Install hivemind itself
    print(f"\n  {BOLD}Installiere HiveMind...{RESET}")
    result = subprocess.run(
        [str(python_path), "-m", "pip", "install", "-e", str(install_dir)],
        capture_output=True, text=True, timeout=120
    )
    if result.returncode != 0:
        print(f"  {YELLOW}⚠ Editable install fehlgeschlagen, versuche normalen Install...{RESET}")
        result = subprocess.run(
            [str(python_path), "-m", "pip", "install", str(install_dir)],
            capture_output=True, text=True, timeout=120
        )
    if result.returncode != 0:
        print(f"  {RED}✗ HiveMind Installation fehlgeschlagen{RESET}")
        return False

    print(f"  {GREEN}✓ HiveMind installiert{RESET}")
    return True


def write_config(install_dir: Path, model_filename: str | None, node_name: str):
    """Schreibe config.yaml."""
    model_path = f"models/{model_filename}" if model_filename else ""
    config = f"""# HiveMind Node Configuration
node:
  name: "{node_name}"
  type: auto

model:
  path: "{model_path}"
  n_ctx: 4096
  n_gpu_layers: -1
  n_threads: 0

cache:
  enabled: true
  max_entries: 10000
  similarity_threshold: 0.92

plugins:
  enabled:
    - chat
    - web_search
  directory: "./plugins"

gateway:
  telegram:
    enabled: false
    token: ""
  api:
    enabled: true
    host: "127.0.0.1"
    port: 8420

network:
  enabled: false
  listen_port: 9420
  bootstrap_nodes: []
"""
    (install_dir / "config.yaml").write_text(config, encoding="utf-8")


def create_launcher(install_dir: Path):
    """Erstelle Start-Skripte für das jeweilige OS."""
    if platform.system() == "Windows":
        # .bat Datei
        bat = install_dir / "HiveMind.bat"
        bat.write_text(f"""@echo off
title HiveMind
cd /d "{install_dir}"
call .venv\\Scripts\\activate.bat
hivemind %*
pause
""", encoding="utf-8")

        # Icon erstellen
        icon_path = install_dir / "hivemind.ico"
        if not icon_path.exists():
            _create_brain_icon(icon_path)

        # Desktop-Verknüpfung
        desktop = Path.home() / "Desktop" / "HiveMind.lnk"
        try:
            ico = str(icon_path).replace("'", "''")
            # Use PowerShell to create shortcut
            ps_cmd = f"""
$s = (New-Object -COM WScript.Shell).CreateShortcut('{desktop}')
$s.TargetPath = '{bat}'
$s.WorkingDirectory = '{install_dir}'
$s.Description = 'HiveMind - Dezentrale P2P-AI'
$s.IconLocation = '{ico},0'
$s.Save()
"""
            subprocess.run(["powershell", "-Command", ps_cmd],
                           capture_output=True, text=True)
            print(f"  {GREEN}✓ Desktop-Verknüpfung erstellt{RESET}")
        except Exception:
            print(f"  {YELLOW}⚠ Desktop-Verknüpfung konnte nicht erstellt werden{RESET}")
            print(f"  {DIM}Starte manuell: {bat}{RESET}")
    else:
        # Shell-Script
        sh = install_dir / "start.sh"
        sh.write_text(f"""#!/bin/bash
cd "{install_dir}"
source .venv/bin/activate
hivemind "$@"
""", encoding="utf-8")
        sh.chmod(0o755)

        # .desktop Datei
        desktop_dir = Path.home() / "Desktop"
        if not desktop_dir.exists():
            desktop_dir = Path.home() / "Schreibtisch"
        if desktop_dir.exists():
            desktop_file = desktop_dir / "HiveMind.desktop"
            desktop_file.write_text(f"""[Desktop Entry]
Version=1.0
Type=Application
Name=🧠 HiveMind
Comment=Dezentrale P2P-AI
Exec=bash -c 'cd "{install_dir}" && source .venv/bin/activate && hivemind'
Icon=utilities-terminal
Terminal=true
Categories=Development;AI;
""", encoding="utf-8")
            desktop_file.chmod(0o755)
            # Trust the desktop file
            subprocess.run(
                ["gio", "set", str(desktop_file), "metadata::trusted", "true"],
                capture_output=True, text=True
            )
            print(f"  {GREEN}✓ Desktop-Verknüpfung erstellt{RESET}")
        else:
            print(f"  {DIM}Starte mit: {sh}{RESET}")


def copy_source(install_dir: Path):
    """Kopiere HiveMind-Quellcode ins Installationsverzeichnis."""
    src_dir = Path(__file__).parent / "hivemind"
    dest_dir = install_dir / "hivemind"

    if src_dir.exists() and src_dir != dest_dir:
        if dest_dir.exists():
            shutil.rmtree(dest_dir)
        shutil.copytree(src_dir, dest_dir)

    # Copy pyproject.toml and other needed files
    for fname in ["pyproject.toml", "README.md", "config.yaml"]:
        src = Path(__file__).parent / fname
        if src.exists():
            dest = install_dir / fname
            if not dest.exists() or src != dest:
                shutil.copy2(src, dest)

    # Create directories
    (install_dir / "models").mkdir(exist_ok=True)
    (install_dir / "plugins").mkdir(exist_ok=True)
    (install_dir / "cache").mkdir(exist_ok=True)


def main():
    banner()

    # ─── Schritt 1: Systemcheck ──────────────────────────────────────
    print(f"{BOLD}Schritt 1/5 — Systemcheck{RESET}\n")
    print(f"  System: {platform.system()} {platform.machine()}")

    python_ok = check_python()
    if not python_ok:
        print(f"\n  {RED}Python {MIN_PYTHON[0]}.{MIN_PYTHON[1]}+ wird benötigt.{RESET}")
        print(f"  {DIM}Download: https://python.org{RESET}")
        sys.exit(1)

    total_ram = check_ram()
    default_dir = get_default_install_dir()
    free_disk = check_disk(default_dir.parent)

    # ─── Schritt 2: Installationsverzeichnis ─────────────────────────
    print(f"\n{BOLD}Schritt 2/5 — Installationsort{RESET}")
    install_dir = Path(ask("Installationsverzeichnis", str(default_dir)))
    install_dir = install_dir.expanduser().resolve()

    if install_dir.exists() and any(install_dir.iterdir()):
        print(f"\n  {YELLOW}⚠ Verzeichnis existiert bereits und ist nicht leer.{RESET}")
        overwrite = ask("Trotzdem fortfahren? (j/n)", "j")
        if overwrite.lower() not in ("j", "ja", "y", "yes"):
            print(f"  {RED}Abgebrochen.{RESET}")
            sys.exit(0)
    
    install_dir.mkdir(parents=True, exist_ok=True)

    # ─── Schritt 3: Modellauswahl ────────────────────────────────────
    print(f"\n{BOLD}Schritt 3/5 — Modell auswählen{RESET}")
    print(f"\n  {DIM}Das AI-Modell ist das 'Gehirn' deines Nodes.")
    print(f"  Größere Modelle = bessere Antworten, aber langsamer + mehr RAM.{RESET}")

    # Filter models based on RAM
    model_options = []
    recommended = 0
    for i, m in enumerate(MODELS):
        if m["url"] is None:
            model_options.append(f"{m['name']}")
            continue
        ram_note = ""
        try:
            ram_needed = float(m["ram"].split()[0])
            if ram_needed > total_ram * 0.8:
                ram_note = f"  {RED}(evtl. zu wenig RAM){RESET}"
        except (ValueError, IndexError):
            pass

        line = f"{m['name']}  |  {m['size']}  |  RAM: {m['ram']}  |  {m['quality']}{ram_note}"
        model_options.append(line)

        # Recommend based on RAM
        try:
            ram_needed = float(m["ram"].split()[0])
            if ram_needed <= total_ram * 0.6:
                recommended = i
        except (ValueError, IndexError):
            pass

    model_idx = ask_choice(
        "Welches Modell möchtest du verwenden?",
        model_options,
        default=recommended + 1
    )
    selected_model = MODELS[model_idx]

    # ─── Schritt 4: Node-Name ────────────────────────────────────────
    print(f"\n{BOLD}Schritt 4/5 — Node einrichten{RESET}")
    default_name = platform.node() or "mein-node"
    node_name = ask("Name für deinen Node", default_name)

    # ─── Zusammenfassung ─────────────────────────────────────────────
    print(f"\n{BOLD}{'─' * 50}{RESET}")
    print(f"  {BOLD}Zusammenfassung:{RESET}")
    print(f"  Verzeichnis: {install_dir}")
    print(f"  Modell:      {selected_model['name']}")
    print(f"  Node-Name:   {node_name}")
    print(f"{'─' * 50}")
    
    confirm = ask("\n  Installation starten? (j/n)", "j")
    if confirm.lower() not in ("j", "ja", "y", "yes"):
        print(f"  {RED}Abgebrochen.{RESET}")
        sys.exit(0)

    # ─── Schritt 5: Installation ─────────────────────────────────────
    print(f"\n{BOLD}Schritt 5/5 — Installation{RESET}\n")

    # Copy source files
    print(f"  {BOLD}Kopiere Dateien...{RESET}")
    copy_source(install_dir)
    print(f"  {GREEN}✓ Dateien kopiert{RESET}")

    # Write config
    write_config(install_dir, selected_model["filename"], node_name)
    print(f"  {GREEN}✓ Konfiguration erstellt{RESET}")

    # Download model
    if selected_model["url"]:
        model_path = install_dir / "models" / selected_model["filename"]
        if model_path.exists():
            print(f"\n  {GREEN}✓ Modell bereits vorhanden: {selected_model['filename']}{RESET}")
        else:
            success = download_file(
                selected_model["url"],
                model_path,
                label=selected_model["name"]
            )
            if not success:
                print(f"  {YELLOW}⚠ Modell konnte nicht heruntergeladen werden.{RESET}")
                print(f"  {DIM}Du kannst es später manuell in {install_dir / 'models'} ablegen.{RESET}")
                write_config(install_dir, None, node_name)

    # Create venv and install
    if not create_venv(install_dir):
        print(f"\n  {RED}✗ Installation fehlgeschlagen.{RESET}")
        print(f"  {DIM}Prüfe die Fehlermeldungen oben und versuche es erneut.{RESET}")
        sys.exit(1)

    # Create launcher
    print(f"\n  {BOLD}Erstelle Starter...{RESET}")
    create_launcher(install_dir)

    # ─── Fertig! ─────────────────────────────────────────────────────
    print(f"""
{GREEN}{BOLD}╔══════════════════════════════════════════════╗
║        ✅ Installation abgeschlossen!         ║
╚══════════════════════════════════════════════╝{RESET}

  {BOLD}So startest du HiveMind:{RESET}
""")
    if platform.system() == "Windows":
        print(f"  • Doppelklick auf {CYAN}HiveMind.bat{RESET} im Installationsordner")
        print(f"  • Oder die {CYAN}Desktop-Verknüpfung{RESET}")
    else:
        print(f"  • Doppelklick auf {CYAN}🧠 HiveMind{RESET} auf dem Desktop")
        print(f"  • Oder im Terminal: {CYAN}cd {install_dir} && ./start.sh{RESET}")

    print(f"""
  {BOLD}Befehle im Chat:{RESET}
  /status   — Node-Status anzeigen
  /plugins  — Plugins auflisten
  /help     — Hilfe
  /quit     — Beenden

  {DIM}Installationsordner: {install_dir}{RESET}
  {DIM}Konfiguration: {install_dir / 'config.yaml'}{RESET}
""")

    # Wait for keypress on Windows
    if platform.system() == "Windows":
        input("  Drücke Enter zum Beenden...")


if __name__ == "__main__":
    main()
