"""Global Memory — persistent facts about the user.

Automatically extracts and stores important information from conversations.
Injected into every session as system context.
"""
from __future__ import annotations

import json
import logging
import re
import time
from pathlib import Path

log = logging.getLogger(__name__)

# Patterns that suggest extractable facts
_FACT_PATTERNS = [
    # Self-introduction
    (r"(?:ich hei[sß]e|mein name ist|i[' ]?m|my name is)\s+(\w+)", "name"),
    (r"(?:ich bin|i am)\s+(\d+)\s*(?:jahre|years)", "alter"),
    (r"(?:ich wohne in|i live in|ich komme aus)\s+(.+?)(?:\.|,|$)", "wohnort"),
    (r"(?:ich arbeite als|i work as|mein beruf ist|ich bin)\s+(\w+(?:\s+\w+)?)\s+(?:von beruf|by profession)", "beruf"),
    (r"(?:meine lieblingsfarbe ist|my favorite color is)\s+(\w+)", "lieblingsfarbe"),
    (r"(?:ich mag|i like|ich liebe|i love)\s+(.+?)(?:\.|,|!|$)", "vorliebe"),
    (r"(?:ich mag kein|i don.?t like|ich hasse|i hate)\s+(.+?)(?:\.|,|!|$)", "abneigung"),
    (r"(?:meine lieblings\w+ ist|my favorite \w+ is)\s+(.+?)(?:\.|,|$)", "favorit"),
    (r"(?:ich spreche|i speak)\s+(.+?)(?:\.|,|$)", "sprachen"),
    (r"(?:nenn mich|call me)\s+(\w+)", "spitzname"),
]


class GlobalMemory:
    """Persistent memory that spans all sessions."""

    def __init__(self, data_dir: str | Path = "data"):
        self.data_dir = Path(data_dir)
        self.data_dir.mkdir(parents=True, exist_ok=True)
        self.memory_path = self.data_dir / "memory.json"
        self._facts: dict[str, str] = {}
        self._custom: list[str] = []
        self._updated = 0.0
        self._load()

    def _load(self):
        if self.memory_path.exists():
            try:
                data = json.loads(self.memory_path.read_text(encoding="utf-8"))
                self._facts = data.get("facts", {})
                self._custom = data.get("custom", [])
                self._updated = data.get("updated", 0)
            except Exception as e:
                log.warning("Failed to load memory: %s", e)

    def _save(self):
        self._updated = time.time()
        data = {
            "facts": self._facts,
            "custom": self._custom,
            "updated": self._updated,
        }
        self.memory_path.write_text(
            json.dumps(data, ensure_ascii=False, indent=2),
            encoding="utf-8",
        )

    def extract_facts(self, text: str) -> list[tuple[str, str]]:
        """Extract facts from user text using pattern matching.
        
        Returns list of (category, value) tuples for newly found facts.
        """
        found = []
        text_lower = text.lower().strip()

        for pattern, category in _FACT_PATTERNS:
            match = re.search(pattern, text_lower, re.IGNORECASE)
            if match:
                value = match.group(1).strip()
                if len(value) > 1 and len(value) < 100:
                    # For list-type facts (vorliebe, abneigung), append
                    if category in ("vorliebe", "abneigung"):
                        existing = self._facts.get(category, "")
                        if value not in existing.lower():
                            new_val = (existing + ", " + value).strip(", ")
                            self._facts[category] = new_val
                            found.append((category, value))
                    else:
                        if self._facts.get(category) != value:
                            self._facts[category] = value
                            found.append((category, value))

        if found:
            self._save()
            for cat, val in found:
                log.info("Memory: %s = %s", cat, val)

        return found

    def process_message(self, role: str, content: str):
        """Process a message and extract any facts from user messages."""
        if role == "user":
            self.extract_facts(content)

    def add_fact(self, category: str, value: str):
        """Manually add a fact."""
        self._facts[category] = value
        self._save()

    def add_custom(self, note: str):
        """Add a custom memory note."""
        if note not in self._custom:
            self._custom.append(note)
            self._save()

    def remove_fact(self, category: str) -> bool:
        if category in self._facts:
            del self._facts[category]
            self._save()
            return True
        return False

    def remove_custom(self, index: int) -> bool:
        if 0 <= index < len(self._custom):
            self._custom.pop(index)
            self._save()
            return True
        return False

    def build_context(self) -> str:
        """Build a context string for the system prompt."""
        if not self._facts and not self._custom:
            return ""

        parts = []
        if self._facts:
            parts.append("Bekannte Informationen ueber den Nutzer:")
            for cat, val in self._facts.items():
                label = _CATEGORY_LABELS.get(cat, cat.capitalize())
                parts.append(f"  - {label}: {val}")

        if self._custom:
            parts.append("Notizen:")
            for note in self._custom:
                parts.append(f"  - {note}")

        return "\\n".join(parts)

    @property
    def all_facts(self) -> dict:
        return dict(self._facts)

    @property
    def all_custom(self) -> list[str]:
        return list(self._custom)

    @property
    def stats(self) -> dict:
        return {
            "facts": len(self._facts),
            "custom": len(self._custom),
            "updated": self._updated,
        }


_CATEGORY_LABELS = {
    "name": "Name",
    "spitzname": "Spitzname",
    "alter": "Alter",
    "wohnort": "Wohnort",
    "beruf": "Beruf",
    "lieblingsfarbe": "Lieblingsfarbe",
    "vorliebe": "Mag",
    "abneigung": "Mag nicht",
    "favorit": "Favorit",
    "sprachen": "Sprachen",
}
