"""Telegram gateway — use Telegram as a thin client for your node."""
from __future__ import annotations

import asyncio
import logging
from typing import Any

log = logging.getLogger(__name__)


class TelegramGateway:
    """Bridges Telegram messages to the local HiveMind node."""

    def __init__(self, token: str, node: Any):
        self.token = token
        self.node = node
        self._app = None

    async def start(self) -> None:
        """Start the Telegram bot."""
        try:
            from telegram import Update
            from telegram.ext import (
                ApplicationBuilder,
                CommandHandler,
                MessageHandler,
                ContextTypes,
                filters,
            )
        except ImportError:
            log.error("python-telegram-bot not installed. Run: pip install 'hivemind[telegram]'")
            return

        self._app = ApplicationBuilder().token(self.token).build()

        async def cmd_start(update: Update, context: ContextTypes.DEFAULT_TYPE):
            await update.message.reply_text(
                f"🧠 HiveMind Node: {self.node.name}\n"
                f"Model: {'loaded' if self.node.model.loaded else 'none'}\n"
                f"Plugins: {', '.join(self.node.plugins.loaded)}\n\n"
                "Sende mir eine Nachricht!"
            )

        async def cmd_status(update: Update, context: ContextTypes.DEFAULT_TYPE):
            s = self.node.status
            await update.message.reply_text(
                f"📊 Status\n"
                f"Node: {s['name']} ({s['id']})\n"
                f"Model: {'✅' if s['model_loaded'] else '❌'}\n"
                f"Plugins: {', '.join(s['plugins'])}\n"
                f"Cache: {s['cache_size']} entries\n"
                f"History: {s['history_length']} messages"
            )

        async def handle_message(update: Update, context: ContextTypes.DEFAULT_TYPE):
            text = update.message.text
            if not text:
                return

            # Show typing indicator
            await update.message.chat.send_action("typing")

            # Process through node
            response = await self.node.chat(text)
            await update.message.reply_text(response)

        self._app.add_handler(CommandHandler("start", cmd_start))
        self._app.add_handler(CommandHandler("status", cmd_status))
        self._app.add_handler(MessageHandler(filters.TEXT & ~filters.COMMAND, handle_message))

        log.info("Telegram gateway starting...")
        await self._app.initialize()
        await self._app.start()
        await self._app.updater.start_polling()
        log.info("Telegram gateway running")

    async def stop(self) -> None:
        if self._app:
            await self._app.updater.stop()
            await self._app.stop()
            await self._app.shutdown()
