"""CLI entry point for HiveMind."""
from __future__ import annotations

import argparse
import asyncio
import logging
import sys
from pathlib import Path

from hivemind.config import Config
from hivemind.node import Node


def setup_logging(verbose: bool = False, log_file: str | None = None) -> None:
    level = logging.DEBUG if verbose else logging.INFO
    handlers: list[logging.Handler] = [logging.StreamHandler()]
    if log_file:
        fh = logging.FileHandler(log_file, encoding="utf-8")
        fh.setLevel(level)
        fh.setFormatter(logging.Formatter(
            "%(asctime)s [%(levelname)s] %(name)s: %(message)s",
            datefmt="%Y-%m-%d %H:%M:%S",
        ))
        handlers.append(fh)
    logging.basicConfig(
        level=level,
        format="%(asctime)s [%(levelname)s] %(name)s: %(message)s",
        datefmt="%H:%M:%S",
        handlers=handlers,
    )


async def run_node(args: argparse.Namespace) -> None:
    """Start the HiveMind node."""
    config = Config.load(args.config)
    base_dir = Path(args.config).parent

    print(f"HiveMind v{__import__('hivemind').__version__}")
    print(f"  Config: {args.config}")
    print(f"  Model:  {config.model.path or '(nicht konfiguriert)'}")
    print()

    node = Node(config, base_dir=base_dir)
    await node.start()

    print(f"\n[OK] Node '{node.name}' bereit!")
    if config.gateway.api.enabled:
        host = config.gateway.api.host
        if host == "0.0.0.0":
            host = "127.0.0.1"
        print(f"  Dashboard: http://{host}:{config.gateway.api.port}")
    print()

    tasks = []

    # Determine private key path for web UI
    private_key_path = base_dir / "hivemind_update_key.private"
    if not private_key_path.exists():
        # Also check parent dirs
        for p in [base_dir.parent, Path.home() / ".openclaw" / "workspace"]:
            candidate = p / "hivemind_update_key.private"
            if candidate.exists():
                private_key_path = candidate
                break

    # Start Web UI + API gateway if enabled
    if config.gateway.api.enabled:
        import uvicorn
        from hivemind.ui.web import create_web_app

        app = create_web_app(node, private_key_path=private_key_path)
        uv_config = uvicorn.Config(
            app,
            host=config.gateway.api.host,
            port=config.gateway.api.port,
            log_level="warning",
        )
        server = uvicorn.Server(uv_config)
        tasks.append(asyncio.create_task(server.serve()))
        logging.getLogger("hivemind").info(
            "Web UI: http://%s:%d", config.gateway.api.host, config.gateway.api.port
        )

    # Start Telegram gateway if enabled
    if config.gateway.telegram.enabled and config.gateway.telegram.token:
        from hivemind.gateway.telegram import TelegramGateway

        tg = TelegramGateway(config.gateway.telegram.token, node)
        tasks.append(asyncio.create_task(tg.start()))

    # Start TUI or daemon mode
    if args.daemon:
        logging.getLogger("hivemind").info("Running in daemon mode (Ctrl+C to stop)")
        # Auto-open dashboard in browser
        if config.gateway.api.enabled:
            import webbrowser
            host = config.gateway.api.host
            if host == "0.0.0.0":
                host = "127.0.0.1"
            url = f"http://{host}:{config.gateway.api.port}"
            try:
                webbrowser.open(url)
            except Exception:
                pass
        try:
            await asyncio.gather(*tasks) if tasks else await asyncio.Event().wait()
        except asyncio.CancelledError:
            pass
    else:
        from hivemind.ui.tui import run_tui
        try:
            await run_tui(node)
        except (EOFError, KeyboardInterrupt):
            pass

    await node.stop()


def open_dashboard(args: argparse.Namespace) -> None:
    """Open the HiveMind dashboard in the default browser."""
    import webbrowser
    config = Config.load(args.config)
    host = config.gateway.api.host
    port = config.gateway.api.port
    if host == "0.0.0.0":
        host = "127.0.0.1"
    url = f"http://{host}:{port}"
    print(f"🧠 Öffne Dashboard: {url}")
    webbrowser.open(url)


def main():
    parser = argparse.ArgumentParser(
        prog="hivemind",
        description="🧠 HiveMind — Dezentrale P2P-AI",
    )
    sub = parser.add_subparsers(dest="command")

    # Default run (no subcommand)
    parser.add_argument(
        "-c", "--config",
        default="config.yaml",
        help="Path to config file (default: config.yaml)",
    )
    parser.add_argument(
        "-v", "--verbose",
        action="store_true",
        help="Enable debug logging",
    )
    parser.add_argument(
        "-d", "--daemon",
        action="store_true",
        help="Run in daemon mode (no TUI, API/Telegram only)",
    )

    # dashboard subcommand
    dash_parser = sub.add_parser("dashboard", help="Dashboard im Browser öffnen")
    dash_parser.add_argument("-c", "--config", default="config.yaml")

    # import-hf subcommand
    hf_parser = sub.add_parser("import-hf", help="HuggingFace Dataset in RAG importieren")
    hf_parser.add_argument("url", help="HuggingFace Dataset URL oder repo-id")
    hf_parser.add_argument("-c", "--config", default="config.yaml")
    hf_parser.add_argument("--max-rows", type=int, default=5000, help="Max Rows (default 5000)")
    hf_parser.add_argument("--split", default="train", help="Dataset Split (default train)")
    hf_parser.add_argument("--fields", nargs="*", help="Text-Felder (auto-detect wenn leer)")

    args = parser.parse_args()
    # Resolve config to absolute path — also check script's own directory
    config_path = Path(args.config)
    if not config_path.exists():
        # Try relative to the hivemind package directory
        pkg_dir = Path(__file__).resolve().parent.parent
        alt = pkg_dir / args.config
        if alt.exists():
            config_path = alt
    args.config = str(config_path.resolve())
    config_dir = Path(args.config).parent
    log_file = str(config_dir / "hivemind.log") if args.command != "dashboard" else None
    # Try to create log file, fall back to no file logging if not writable
    if log_file:
        try:
            Path(log_file).touch(exist_ok=True)
        except (PermissionError, OSError):
            log_file = None
    setup_logging(getattr(args, "verbose", False), log_file=log_file)

    if args.command == "dashboard":
        open_dashboard(args)
        return

    if args.command == "import-hf":
        from hivemind.rag import RAGStore
        config = Config.load(args.config)
        base_dir = Path(args.config).parent
        rag = RAGStore(data_dir=base_dir / config.rag.data_dir, chunk_size=config.rag.chunk_size)
        print(f"Importiere HuggingFace Dataset: {args.url}")
        print(f"  Split: {args.split} | Max Rows: {args.max_rows}")
        result = rag.import_huggingface(args.url, args.max_rows, args.fields, args.split)
        if "error" in result:
            print(f"  FEHLER: {result['error']}")
            sys.exit(1)
        print(f"  Rows importiert: {result['rows_imported']} / {result['rows_total']}")
        print(f"  Text-Felder: {', '.join(result['text_fields'])}")
        print(f"  Chunks: {result['chunks']}")
        print(f"  Groesse: {result['size_mb']} MB")
        print(f"  Dokument: {result['document']}")
        print("Fertig!")
        return

    try:
        asyncio.run(run_node(args))
    except KeyboardInterrupt:
        print("\nHiveMind stopped.")


if __name__ == "__main__":
    main()
