"""Date/time awareness plugin — gibt dem Node Kenntnis über aktuelles Datum und Uhrzeit."""
from __future__ import annotations

import logging
from datetime import datetime, timezone

from hivemind.plugins.base import Plugin, capability

log = logging.getLogger(__name__)

WEEKDAYS_DE = [
    "Montag", "Dienstag", "Mittwoch", "Donnerstag",
    "Freitag", "Samstag", "Sonntag",
]
MONTHS_DE = [
    "Januar", "Februar", "März", "April", "Mai", "Juni",
    "Juli", "August", "September", "Oktober", "November", "Dezember",
]


class DateTimePlugin(Plugin):
    """Plugin that provides current date/time awareness."""

    name = "datetime_info"
    version = "0.1.0"
    description = "Provides current date and time awareness to the node"

    @capability("Get the current date, time, weekday and timezone")
    async def get_datetime(self, tz_name: str = "") -> str:
        """Return a human-readable current date/time string.

        Args:
            tz_name: IANA timezone name (e.g. 'Europe/Berlin').
                     Leave empty to use the system's local timezone.
        """
        try:
            if tz_name:
                try:
                    import zoneinfo
                    tz = zoneinfo.ZoneInfo(tz_name)
                    now = datetime.now(tz)
                except Exception:
                    now = datetime.now().astimezone()
            else:
                now = datetime.now().astimezone()  # local timezone

            weekday = WEEKDAYS_DE[now.weekday()]
            month = MONTHS_DE[now.month - 1]
            tz_str = now.strftime("%Z")

            return (
                f"{weekday}, {now.day}. {month} {now.year}, "
                f"{now.strftime('%H:%M')} Uhr {tz_str}"
            )
        except Exception as e:
            log.error("DateTimePlugin.get_datetime error: %s", e)
            now = datetime.now(timezone.utc)
            return now.strftime("UTC: %Y-%m-%d %H:%M")

    @capability("Get a compact date/time string for embedding in the system prompt")
    async def context_string(self) -> str:
        """Return a one-liner to inject into the system prompt so the model
        knows the current point in time without being asked."""
        dt = await self.get_datetime()
        return f"Aktuelles Datum und Uhrzeit: {dt}"


PLUGIN_CLASS = DateTimePlugin
