"""Tagebuch-Hilfsfunktionen — systemweites Event-Logging.

Dieses Modul stellt `diary_append` bereit, das aus beliebigen Teilen des
Systems (node.py, network/peer.py, usw.) aufgerufen werden kann, ohne eine
zirkuläre Abhängigkeit zu create_web_app() einzugehen.
"""
from __future__ import annotations

import datetime
import json
from pathlib import Path


def diary_append(base_dir: Path, source: str, agent: str, entry: str) -> None:
    """Hängt einen kurzen Tagebucheintrag an die heutige Tagesdatei an.

    Parameters
    ----------
    base_dir:  Basisverzeichnis des Nodes (``node.base_dir``)
    source:    Quelle/Kategorie des Eintrags (z. B. ``"system"``, ``"network"``, ``"plugin"``)
    agent:     Name des Agenten / Nodes der den Eintrag erzeugt
    entry:     Eintragstext (wird bei leerem Wert ignoriert)
    """
    if not (entry or "").strip():
        return
    base_dir = Path(base_dir)
    diary_dir = base_dir / "data" / "diary"
    try:
        diary_dir.mkdir(parents=True, exist_ok=True)
    except Exception:
        return
    path = diary_dir / f"diary_{datetime.date.today().isoformat()}.json"
    entries: list = []
    if path.exists():
        try:
            entries = json.loads(path.read_text(encoding="utf-8")) or []
        except Exception:
            entries = []
    entries.append({
        "ts": datetime.datetime.now().strftime("%H:%M:%S"),
        "source": source,
        "agent": agent,
        "entry": entry,
    })
    try:
        path.write_text(json.dumps(entries, ensure_ascii=False, indent=2), encoding="utf-8")
    except Exception:
        pass
